﻿/*================================================================================*/
/*
**	PROJECT:	ARL-AG8-LC (A)
**	VERSION:	AG1.10
**	AUTHOR:  	Uma Arthika Katikapalli
**	COMPANY: 	Arlyn Scales, Inc.
**			 	59 2nd St.,
**			 	East Rockaway, NY 11518
**	DATE:		05/31/2018
*/
/***********************************************************************************/

using System;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;

using FTD2XX_NET;
using System.IO;
using System.Threading;

namespace CSharp_FT4222_SPI_Master
{
    static class Constants
    {
        public const byte CHANNELS = 8;                             // Number of channels in AD7770 = 8

        // User configuration register to control mode of working of AD7770
        public const byte AD7770_REG_GENERAL_USER_CONFIG_3 = 0x13;                      // Register Address of GENERAL_USER_CONFIG_3 for AD7770
        public const byte GENERAL_USER_CONFIG_3_DEFAULT = 0x80;                         // Default Register value stored in GENERAL_USER_CONFIG_3 with SPI_SLAVE_MODE_EN bit at Bit 4 
        public const byte SPI_SLAVE_MODE_EN = 0x10 | GENERAL_USER_CONFIG_3_DEFAULT;     // Value to be updated in GENERAL_USER_CONFIG_3 to begin analog to digital conversions, i.e. setting Bit 4 

        public const byte AD7770_REG_GENERAL_USER_CONFIG_1 = 0x11;                      // Register Address of GENERAL_USER_CONFIG_1 for AD7770
        public const byte GENERAL_USER_CONFIG_1_DEFAULT = 0x24;                         // Reset value for the register
        public const byte HIGH_RESOLUTION = 0x40 | GENERAL_USER_CONFIG_1_DEFAULT;       // Set bit 6 of GENERAL_USER_CONFIG_1 to enable High Resolution mode

        // Decimation registers used to configure sampling rate
        public const byte AD7770_REG_SRC_N_MSB = 0x60;				// Register Address of Decimation Rate (N) MSB
        public const byte AD7770_REG_SRC_N_LSB = 0x61;				// Register Address of Decimation Rate (N) LSB
        public const byte AD7770_REG_SRC_IF_MSB = 0x62;				// Register Address of Decimation Rate (IF) MSB
        public const byte AD7770_REG_SRC_IF_LSB = 0x63;             // Register Address of Decimation Rate (IF) LSB

        public const byte AD7770_REG_SRC_UPDATE = 0x64;             // SRC UPDATE
        public const byte SRC_LOAD_UPDATE = 0x01;                   // Bit[0] of SRC_UPDATE register to load SRC registers into SRC

        public const byte PGA_4 = 0x80;                             // Register value is 0x80, meaning Programmable Gain Amplifier (PGA) = 4 
        public const byte AD7770_CH0_CONFIG = 0x00;	                // Register address of Channel 0 Configuration

        public const byte AD7770_GENERAL_ERRORS_1 = 0x59;           // Register address of General Errors Register 1
    }

    class Program
    {
        //**************************************************************************
        //
        // FUNCTION IMPORTS FROM FTD2XX DLL
        //
        //**************************************************************************

        /**************************************************************************
        Function Name:  FT_CreateDeviceInfoList
        Parameters:     Pointer to unsigned long to store number of devices connected
        Description:    This function builds a device information list and returns 
                        the number of D2XX devices connected to the system. This list 
                        includes both open and unopen devices.
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("ftd2xx.dll")]
        static extern FTDI.FT_STATUS FT_CreateDeviceInfoList(ref UInt32 numdevs);

        /**************************************************************************
        Function Name:  FT_GetDeviceInfoDetail
        Parameters:     index - index of the entry in the device info list,
                        flags - pointer to store the flag value
                        chiptype - pointer to store device type
                        id - pointer to store device id
                        locid - pointer to store location id
                        serialnumber - pointer to store device serial number
                        description - pointer to store device description 
                        ftHandle - pointer where handle is stored
        Description:    This function returns an entry from the Device information 
                        list
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("ftd2xx.dll")]
        static extern FTDI.FT_STATUS FT_GetDeviceInfoDetail(UInt32 index, ref UInt32 flags, ref FTDI.FT_DEVICE chiptype, ref UInt32 id, ref UInt32 locid, byte[] serialnumber, byte[] description, ref IntPtr ftHandle);

        /**************************************************************************
        Function Name:  FT_OpenEx
        Parameters:     pvArg1 - pointer to a null terminated string, type depends on dwFlags
                        dwFlags - FT_OPEN_BY_SERIAL_NUMBER, FT_OPEN_BY_DESCRIPTION, 
                                  FT_OPEN_BY_LOCATION
                        ftHandle - pointer where handle is stored
        Description:    Opens the specified device and return a handle that will be used for 
                        subsequent access. The device can be specified by its serial number, 
                        device description or location
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        //[DllImportAttribute("ftd2xx.dll", CallingConvention = CallingConvention.Cdecl)]
        [DllImport("ftd2xx.dll")]
        static extern FTDI.FT_STATUS FT_OpenEx(string pvArg1, int dwFlags, ref IntPtr ftHandle);

        /**************************************************************************
        Function Name:  FT_Close
        Parameters:     ftHandle - handle of the device
        Description:    Coses an open device and releases its resources
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        //[DllImportAttribute("ftd2xx.dll", CallingConvention = CallingConvention.Cdecl)]
        [DllImport("ftd2xx.dll")]
        static extern FTDI.FT_STATUS FT_Close(IntPtr ftHandle);

        // Values for FT_Open
        const byte FT_OPEN_BY_SERIAL_NUMBER = 1;
        const byte FT_OPEN_BY_DESCRIPTION = 2;
        const byte FT_OPEN_BY_LOCATION = 4;
        const string FT4222A = "FT4222 A";
        const string FT4222B = "FT4222 B";

        //**************************************************************************
        //
        // FUNCTION IMPORTS FROM LIBFT4222 DLL
        //
        //**************************************************************************

        /**************************************************************************
        Function Name:  FT4222_SetClock
        Parameters:     ftHandle - handle of the device
                        clk - system clock rate
        Description:    Set the system clock rate. The FT4222H supports 4 clock 
                        rates: 80MHz, 60MHz, 48MHz, or 24MHz.By default, the FT4222H 
                        runs at 60MHz clock rate.
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SetClock(IntPtr ftHandle, FT4222_ClockRate clk);

        /**************************************************************************
        Function Name:  FT4222_GetClock
        Parameters:     ftHandle - handle of the device
                        clk - pointer of type FT4222_ClockRate to where the value 
                            will be stored
        Description:    Get the current system clock   
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_GetClock(IntPtr ftHandle, ref FT4222_ClockRate clk);

        /**************************************************************************
        Function Name:  FT4222_SPIMaster_Init
        Parameters:     ftHandle - handle of the device
                        ioLine - single, dual or quad
                        clock - clock divider
                        cpol - clock polarity
                        cpha - clock phase
                        ssoMap - slave selection output pin (bitmap)
        Description:    Initialize the FT4222H USB chip as the SPI master
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SPIMaster_Init(IntPtr ftHandle, FT4222_SPIMode ioLine, FT4222_SPIClock clock, FT4222_SPICPOL cpol, FT4222_SPICPHA cpha, Byte ssoMap);

        /**************************************************************************
        Function Name:  FT4222_SPI_SetDrivingStrength
        Parameters:     ftHandle - handle of the device
                        clkStrength - The driving strength of the clk pin (SPI master)
                        ioStrength - The driving strength of the io pin
                        ssoStrength - The driving strength of the sso pin (SPI master)
        Description:    For the FT4222H SPI, set the driving strength of clk, io, and
                        sso pins
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SPI_SetDrivingStrength(IntPtr ftHandle, SPI_DrivingStrength clkStrength, SPI_DrivingStrength ioStrength, SPI_DrivingStrength ssoStregth);

        /**************************************************************************
        Function Name:  FT4222_SPIMaster_SingleReadWrite
        Parameters:     ftHandle - handle of the device
                        readBuffer - Pointer to the buffer that receives data from 
                                    the device
                        writeBuffer - Pointer to the buffer that contains data to be 
                                    written to the device
                        bufferSize - The size of read and write buffer, same size
                        sizeTranferred - Pointer to a variable which receives the 
                                    number of bytes read and written to the device
                        isEndTransaction - TRUE to raise the pin of SS at the end of 
                                    the transaction
        Description:    Under SPI single mode, full-duplex write data to and read 
                        data from an SPI slave. The standard SPI protocol simultaneously 
                        sends data onto the MOSI data line and receives data from 
                        the MISO line 
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SPIMaster_SingleReadWrite(IntPtr ftHandle, ref byte readBuffer, ref byte writeBuffer, ushort bufferSize, ref ushort sizeTransferred, bool isEndTransaction);

        /**************************************************************************
        Function Name:  FT4222_SPIMaster_SingleWrite
        Parameters:     ftHandle - handle of the device
                        buffer - Pointer to the buffer that contains data to be 
                                    written to the device
                        bufferSize - Number of bytes to write to the device
                        sizeTranferred - Pointer to a variable which receives the 
                                    number of bytes written to the device
                        isEndTransaction - TRUE to raise the pin of SS at the end of 
                                    the write
        Description:    Under SPI single mode, write data to an SPI slave
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SPIMaster_SingleWrite(IntPtr ftHandle, ref byte buffer, ushort bufferSize, ref ushort sizeTransferred, bool isEndTransaction);

        /**************************************************************************
        Function Name:  FT4222_SPIMaster_SingleRead
        Parameters:     ftHandle - handle of the device
                        buffer - Pointer to the buffer that receives the data from 
                                the device
                        bufferSize - Number of bytes to read from the device
                        sizeTranferred - Pointer to a variable which receives the 
                                    number of bytes read from the device
                        isEndTransaction - TRUE to raise the pin of SS at the end of 
                                    the write
        Description:    Under SPI single mode, read data from an SPI slave.
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SPIMaster_SingleRead(IntPtr ftHandle, ref byte buffer, ushort bufferSize, ref ushort sizeTransferred, bool isEndTransaction);


        /**************************************************************************
        Function Name:  FT4222_GPIO_Init
        Parameters:     ftHandle - handle of the device
                        gpio_Dir - An array defines the directions of 4 GPIO pins.
                               The GPIO direction will be GPIO_OUTPUT or GPIO_INPUT
        Description:    Initialize the GPIO interface of the FT4222H.
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_GPIO_Init(IntPtr ftHandle, ref GPIO_Dir gpio_Dir);

        /**************************************************************************
        Function Name:  FT4222_GPIO_Read
        Parameters:     ftHandle - handle of the device
                        portNum - A One of the following GPIO ports:
                        pValue - GPIO function: TRUE means high, FALSE means low
                                 Interrupt: TRUE means trigger condition in invoked
                                       FALSE means trigger condition is not invoked
        Description:    Read the status of a specified GPIO pin or interrupt register
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_GPIO_Read(IntPtr ftHandle, GPIO_Port portNum, ref bool pValue);

        /**************************************************************************
        Function Name:  FT4222_GPIO_Write
        Parameters:     ftHandle - handle of the device
                        portNum - A One of the GPIO ports:
                        pValue - GPIO function: TRUE means high, FALSE means low
        Description:    Write HIGH or LOW to a specified GPIO pin
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_GPIO_Write(IntPtr ftHandle, GPIO_Port portNum, bool pValue);

        /**************************************************************************
        Function Name:  FT4222_SetWakeUpInterrupt
        Parameters:     ftHandle - handle of the device
                        pValue - TRUE to configure GPIO3 as an input pin for 
                             wakeup/interrupt.FALSE to switch back to GPIO3.
        Description:    When Wake up/Interrupt function is on, GPIO3 pin acts as an 
                        input pin for wakeup/interrupt.
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SetWakeUpInterrupt(IntPtr ftHandle, bool pValue);

        /**************************************************************************
        Function Name:  FT4222_SetSuspendOut
        Parameters:     ftHandle - handle of the device
                        pValue - TRUE to enable suspend out and configure GPIO2 as 
                             an output pin for emitting a signal when suspended. 
                             FALSE to switch back to GPIO2.
        Return Value:   FT_OK returned if successful, else FT error code returned
        **************************************************************************/
        [DllImport("LibFT4222.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern FT4222_STATUS FT4222_SetSuspendOut(IntPtr ftHandle, bool pValue);





        // GPIO initialization
        public enum GPIO_Port
        {
            GPIO_PORT0 = 0,
            GPIO_PORT1,
            GPIO_PORT2,
            GPIO_PORT3
        };

        public enum GPIO_Dir
        {
            GPIO_OUTPUT = 0,
            GPIO_INPUT
        };

        public enum GPIO_Trigger
        {
            GPIO_TRIGGER_RISING = 0x01,
            GPIO_TRIGGER_FALLING = 0x02,
            GPIO_TRIGGER_LEVEL_HIGH = 0x04,
            GPIO_TRIGGER_LEVEL_LOW = 0X08
        };

        public enum GPIO_Output
        {
            GPIO_OUTPUT_LOW,
            GPIO_OUTPUT_HIGH
        };

        // FT4222 Device status
        public enum FT4222_STATUS
        {
            FT4222_OK,
            FT4222_INVALID_HANDLE,
            FT4222_DEVICE_NOT_FOUND,
            FT4222_DEVICE_NOT_OPENED,
            FT4222_IO_ERROR,
            FT4222_INSUFFICIENT_RESOURCES,
            FT4222_INVALID_PARAMETER,
            FT4222_INVALID_BAUD_RATE,
            FT4222_DEVICE_NOT_OPENED_FOR_ERASE,
            FT4222_DEVICE_NOT_OPENED_FOR_WRITE,
            FT4222_FAILED_TO_WRITE_DEVICE,
            FT4222_EEPROM_READ_FAILED,
            FT4222_EEPROM_WRITE_FAILED,
            FT4222_EEPROM_ERASE_FAILED,
            FT4222_EEPROM_NOT_PRESENT,
            FT4222_EEPROM_NOT_PROGRAMMED,
            FT4222_INVALID_ARGS,
            FT4222_NOT_SUPPORTED,
            FT4222_OTHER_ERROR,
            FT4222_DEVICE_LIST_NOT_READY,

            FT4222_DEVICE_NOT_SUPPORTED = 1000,        // FT_STATUS extending message
            FT4222_CLK_NOT_SUPPORTED,                  // spi master do not support 80MHz/CLK_2
            FT4222_VENDER_CMD_NOT_SUPPORTED,
            FT4222_IS_NOT_SPI_MODE,
            FT4222_IS_NOT_I2C_MODE,
            FT4222_IS_NOT_SPI_SINGLE_MODE,
            FT4222_IS_NOT_SPI_MULTI_MODE,
            FT4222_WRONG_I2C_ADDR,
            FT4222_INVAILD_FUNCTION,
            FT4222_INVALID_POINTER,
            FT4222_EXCEEDED_MAX_TRANSFER_SIZE,
            FT4222_FAILED_TO_READ_DEVICE,
            FT4222_I2C_NOT_SUPPORTED_IN_THIS_MODE,
            FT4222_GPIO_NOT_SUPPORTED_IN_THIS_MODE,
            FT4222_GPIO_EXCEEDED_MAX_PORTNUM,
            FT4222_GPIO_WRITE_NOT_SUPPORTED,
            FT4222_GPIO_PULLUP_INVALID_IN_INPUTMODE,
            FT4222_GPIO_PULLDOWN_INVALID_IN_INPUTMODE,
            FT4222_GPIO_OPENDRAIN_INVALID_IN_OUTPUTMODE,
            FT4222_INTERRUPT_NOT_SUPPORTED,
            FT4222_GPIO_INPUT_NOT_SUPPORTED,
            FT4222_EVENT_NOT_SUPPORTED,
        };

        // System clcok rates
        public enum FT4222_ClockRate
        {
            SYS_CLK_60 = 0,
            SYS_CLK_24,
            SYS_CLK_48,
            SYS_CLK_80,

        };

        // SPI Mode of operation
        public enum FT4222_SPIMode
        {
            SPI_IO_NONE = 0,
            SPI_IO_SINGLE = 1,
            SPI_IO_DUAL = 2,
            SPI_IO_QUAD = 4,

        };

        // SPI clock (divides the system clock)
        public enum FT4222_SPIClock
        {
            CLK_NONE = 0,
            CLK_DIV_2,      // 1/2   System Clock
            CLK_DIV_4,      // 1/4   System Clock
            CLK_DIV_8,      // 1/8   System Clock
            CLK_DIV_16,     // 1/16  System Clock
            CLK_DIV_32,     // 1/32  System Clock
            CLK_DIV_64,     // 1/64  System Clock
            CLK_DIV_128,    // 1/128 System Clock
            CLK_DIV_256,    // 1/256 System Clock
            CLK_DIV_512,    // 1/512 System Clock

        };

        // SPI clock polarity
        public enum FT4222_SPICPOL
        {
            CLK_IDLE_LOW = 0,
            CLK_IDLE_HIGH = 1,
        };

        // SPI clock phase
        public enum FT4222_SPICPHA
        {
            CLK_LEADING = 0,
            CLK_TRAILING = 1,
        };

        // SPI slave output driving strength
        public enum SPI_DrivingStrength
        {
            DS_4MA = 0,
            DS_8MA,
            DS_12MA,
            DS_16MA,
        };

        static ManualResetEvent manualResetEvent = new ManualResetEvent(false);

        static void Main(string[] args)
        {
            // Declarations
            FTDI.FT_DEVICE_INFO_NODE devInfo = new FTDI.FT_DEVICE_INFO_NODE();
            IntPtr ftHandle_SPI = new IntPtr();
            IntPtr ftHandle_GPIO = new IntPtr();
            FTDI.FT_STATUS ftStatus = 0;
            FT4222_STATUS ft42Status = 0;
            int openClose = 0;
            int attempts;       // number of attempts to know if GPIO read for DRDY pulses failed

            open_close:

            attempts = 5;
            // devices connected to the system
            UInt32 numOfDevices = 0;
            ftStatus = FT_CreateDeviceInfoList(ref numOfDevices);
            if (numOfDevices > 0)
            {
                byte[] sernum = new byte[16];
                byte[] desc = new byte[64];

                ftStatus = FT_GetDeviceInfoDetail(0, ref devInfo.Flags, ref devInfo.Type, ref devInfo.ID, ref devInfo.LocId,
                                            sernum, desc, ref devInfo.ftHandle);

                devInfo.SerialNumber = Encoding.ASCII.GetString(sernum, 0, 16);
                devInfo.Description = Encoding.ASCII.GetString(desc, 0, 64);
                devInfo.SerialNumber = devInfo.SerialNumber.Substring(0, devInfo.SerialNumber.IndexOf("\0"));
                devInfo.Description = devInfo.Description.Substring(0, devInfo.Description.IndexOf("\0"));
            }
            else
            {
                Console.WriteLine("No device connected");
                Console.WriteLine("NG! Press Enter to continue.");
                Console.ReadLine();
                return;
            }


            // Open device
            ftStatus = FT_OpenEx(FT4222A, FT_OPEN_BY_DESCRIPTION, ref ftHandle_SPI);
            if (ftStatus != FTDI.FT_STATUS.FT_OK)
            {
                Console.WriteLine("Open NG: {0}", ftStatus);
                Console.WriteLine("NG! Press Enter to continue.");
                Console.ReadLine();
                return;
            }

            System.Threading.Thread.Sleep(100);

            ftStatus = FT_OpenEx(FT4222B, FT_OPEN_BY_DESCRIPTION, ref ftHandle_GPIO);
            if (ftStatus != FTDI.FT_STATUS.FT_OK)
            {
                Console.WriteLine("Open NG: {0}", ftStatus);
                Console.WriteLine("NG! Press Enter to continue.");
                Console.ReadLine();
                return;
            }

            System.Threading.Thread.Sleep(100);


            GPIO_Dir[] gpioDir = new GPIO_Dir[4];
            gpioDir[0] = GPIO_Dir.GPIO_INPUT;
            gpioDir[1] = GPIO_Dir.GPIO_INPUT;
            gpioDir[2] = GPIO_Dir.GPIO_OUTPUT;
            gpioDir[3] = GPIO_Dir.GPIO_INPUT;

            FT4222_SetSuspendOut(ftHandle_GPIO, false);
            FT4222_SetWakeUpInterrupt(ftHandle_GPIO, true);
            FT4222_GPIO_Init(ftHandle_GPIO, ref gpioDir[0]);

            System.Threading.Thread.Sleep(100);
            bool value = false;


            // RESET Line send pulse
            FT4222_GPIO_Write(ftHandle_GPIO, GPIO_Port.GPIO_PORT2, true);
            System.Threading.Thread.Sleep(300);  //200
            FT4222_GPIO_Write(ftHandle_GPIO, GPIO_Port.GPIO_PORT2, false);
            System.Threading.Thread.Sleep(300);
            FT4222_GPIO_Write(ftHandle_GPIO, GPIO_Port.GPIO_PORT2, true);
            System.Threading.Thread.Sleep(100);

            // Set FT4222 system clock
            FT4222_ClockRate ft4222_Clock = FT4222_ClockRate.SYS_CLK_80;
            ft42Status = FT4222_SetClock(ftHandle_SPI, FT4222_ClockRate.SYS_CLK_80);
            if (ft42Status != FT4222_STATUS.FT4222_OK)
            {
                Console.WriteLine("SetClock NG: {0}. Press Enter to continue.", ft42Status);
                Console.ReadLine();
                return;
            }
            else
            {
                ft42Status = FT4222_GetClock(ftHandle_SPI, ref ft4222_Clock);
                if (ft42Status != FT4222_STATUS.FT4222_OK)
                {
                    Console.WriteLine("GetClock NG: {0}. Press Enter to continue.", ft42Status);
                    Console.ReadLine();
                    return;
                }
            }

            ft42Status = FT4222_SetClock(ftHandle_GPIO, FT4222_ClockRate.SYS_CLK_48);

            // Initialize FT4222 as SPI Master
            ft42Status = FT4222_SPIMaster_Init(ftHandle_SPI, FT4222_SPIMode.SPI_IO_SINGLE, FT4222_SPIClock.CLK_DIV_2, FT4222_SPICPOL.CLK_IDLE_LOW, FT4222_SPICPHA.CLK_LEADING, 0x01);
            if (ft42Status != FT4222_STATUS.FT4222_OK)
            {
                Console.WriteLine("Open NG: {0}", ft42Status);
                Console.WriteLine("NG! Press Enter to continue.");
                Console.ReadLine();
                return;
            }


            // Set driving strength
            ft42Status = FT4222_SPI_SetDrivingStrength(ftHandle_SPI, SPI_DrivingStrength.DS_12MA, SPI_DrivingStrength.DS_12MA, SPI_DrivingStrength.DS_16MA);
            if (ft42Status!= FT4222_STATUS.FT4222_OK)
            {
                Console.WriteLine("SetDrivingStrength NG: {0}", ft42Status);
                Console.WriteLine("NG! Press Enter to continue.");
                Console.ReadLine();
                return;
            }


            // Define variables to store values for register reads and writes
            ushort sizeTransferred = 0;
            byte[] rxBuf = new byte[2] { 0x00, 0x00 };
            byte[] txBuf = new byte[2];
            byte[] txBuf1 = new byte[2];


            // Configure the SPI to register R/W mode
            txBuf[0] = (byte)(Constants.AD7770_REG_GENERAL_USER_CONFIG_3);
            txBuf[1] = (byte)(Constants.GENERAL_USER_CONFIG_3_DEFAULT);
            ft42Status = FT4222_SPIMaster_SingleWrite(ftHandle_SPI, ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);


            // Configure ADC to High Resolution mode
            txBuf[0] = (byte)(Constants.AD7770_REG_GENERAL_USER_CONFIG_1);
            txBuf[1] = (byte)(Constants.HIGH_RESOLUTION);
            ft42Status = FT4222_SPIMaster_SingleWrite(ftHandle_SPI, ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);


            // Configuring Programmable Gain Amplifier to 4 for all channels
            for (int i = Constants.CHANNELS - 1; i >= 0; i--)
            {
                txBuf[0] = (byte)(0x00 | (i - Constants.AD7770_CH0_CONFIG));
                txBuf[1] = (byte)(0x00 | Constants.PGA_4);
                ft42Status = FT4222_SPIMaster_SingleWrite(ftHandle_SPI, ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);
            }


            // Configuring the decimation registers to have a sampling rate of 1000SPS
            // Writng the MSB of the N value into the register
            txBuf[0] = (byte)(0x00 | Constants.AD7770_REG_SRC_N_MSB);
            txBuf[1] = (byte)(0x08);
            ft42Status = FT4222_SPIMaster_SingleReadWrite(ftHandle_SPI, ref rxBuf[0], ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);
            // Writng the LSB of the N value into the register
            txBuf[0] = (byte)(0x00 | Constants.AD7770_REG_SRC_N_LSB);
            txBuf[1] = (byte)(0x00);
            ft42Status = FT4222_SPIMaster_SingleReadWrite(ftHandle_SPI, ref rxBuf[0], ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);
            // Writng the MSB of the IF value into the register
            txBuf[0] = (byte)(0x00 | Constants.AD7770_REG_SRC_IF_MSB);
            txBuf[1] = (byte)(0x00);
            ft42Status = FT4222_SPIMaster_SingleReadWrite(ftHandle_SPI, ref rxBuf[0], ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);
            // Writng the LSB of the IF value into the register
            txBuf[0] = (byte)(0x00 | Constants.AD7770_REG_SRC_IF_LSB);
            txBuf[1] = (byte)(0x00);
            ft42Status = FT4222_SPIMaster_SingleReadWrite(ftHandle_SPI, ref rxBuf[0], ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);


            // SRC UPDATE
            txBuf[0] = (byte)(0x00 | Constants.AD7770_REG_SRC_UPDATE);
            txBuf[1] = (byte)(Constants.SRC_LOAD_UPDATE);
            ft42Status = FT4222_SPIMaster_SingleReadWrite(ftHandle_SPI, ref rxBuf[0], ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);
            System.Threading.Thread.Sleep(5);
            // Resetting the SRC_LOAD_UPDATE bit
            txBuf[0] = (byte)(0x00 | Constants.AD7770_REG_SRC_UPDATE);
            txBuf[1] = (byte)(0x00);
            ft42Status = FT4222_SPIMaster_SingleReadWrite(ftHandle_SPI, ref rxBuf[0], ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);


            int bytesize = 32;
            byte[] rxADCrawData1 = new byte[bytesize];                  // The bytes are each read of 32 bytes are stored in this buffer
            byte[] tx = new byte[bytesize];
            int[] ADCrawData = new int[8];             // ADC readings stored


            // The value of the read back command, the size of the reach back command must be the same the size of bytes to be read
            tx[0] = (byte)(0x80);


            // Configure the SPI slave enable bit in General User Config 3 register to read back ADC data
            txBuf[0] = (byte)(0x00 | 0x13);
            txBuf[1] = (byte)(0x00 | 0x90);              // ADC data conversion mode         
            ft42Status = FT4222_SPIMaster_SingleWrite(ftHandle_SPI, ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);
            System.Threading.Thread.Sleep(40);


            if (openClose != 0)
            {
                // Configure the SPI to register R/W mode
                txBuf[0] = (byte)(Constants.AD7770_REG_GENERAL_USER_CONFIG_3);
                txBuf[1] = (byte)(Constants.GENERAL_USER_CONFIG_3_DEFAULT);
                ft42Status = FT4222_SPIMaster_SingleWrite(ftHandle_SPI, ref txBuf[0], (ushort)txBuf.Length, ref sizeTransferred, true);

                openClose--;
                System.Threading.Thread.Sleep(100);
                FT_Close(ftHandle_SPI);
                FT_Close(ftHandle_GPIO);
                System.Threading.Thread.Sleep(20);
                goto open_close;
            }


            System.Threading.Thread.Sleep(400);
            Console.WriteLine("READY! Converting ADC data... ");


            // channel 0 needs to be configured for PGA = 4 (to prevent overwriting the AD7770_CH0_CONFIG register with PGA = 1 because of the read back command)
            txBuf1[0] = (byte)(0x00 | (Constants.AD7770_CH0_CONFIG));
            txBuf1[1] = (byte)(0x00 | Constants.PGA_4);

            System.Threading.Thread.Sleep(20);


            // testing variables and markers
            double TotalSampleCount = 0,        // total samples counter
                TS1 = 0,                        // total samples marker at the approx beginning of the batch of mismatches
                TS2 = 0,                        // total samples marker at the approx end of the batch of bad reads
                SamplesDuringBatch = 0;         // difference between the above two markers to get total reads during the batch of mismatches
            double ChannelIDError = 0,          // total channel mismatch counter
                ChM1 = 0,                       // marker to help enumerate mismatches within the batch 
                ChM2 = 0,                       // marker to help enumerate mismatches within the batch     
                mismatchBatchOccurrence = 0;    // mismatch batch counter over duration of run
            bool mismatchBatch = false;         // mismatch batch flag
            double OutOfRangeError = 0;         // out of ranger error counter
            double efficiency = 0.0;            // over all good sample %
            double batch_eff = 0;               // % of samples that are good during the batch of bad reads

            Stopwatch stopwatch = new Stopwatch();      // time stamping
            stopwatch.Start();

            TimeSpan mismatch_mark1 = stopwatch.Elapsed,    // timestamp to help indicate if the batch of mismatch begins
                mismatch_mark2,                             // timestamp to help indicate if the batch of mismatch begins
                mismatch_batch1 = stopwatch.Elapsed,        // marker at the approx beginning fo the mismatch batch
                mismatch_batch2 = stopwatch.Elapsed;        // marker at the approx end fo the mismatch batch
            double batch_period = 0;                        // this variable stores the difference in the above two markers (gives approx time duration of the batch of mismacthes)

            while (true)
            {
                read_data:
                if (FT4222_GPIO_Read(ftHandle_GPIO, GPIO_Port.GPIO_PORT3, ref value) == FT4222_STATUS.FT4222_OK)            // added for extra sync when mismatch occurs
                {
                    if (attempts == 0)
                    {
                        FT_Close(ftHandle_SPI);
                        FT_Close(ftHandle_GPIO);
                        Console.CursorTop = 0;
                        System.Threading.Thread.Sleep(4000);
                        goto open_close;
                    }
                    if (value)
                    {
                        value = false;
                        System.Threading.Thread.Sleep(2);
                        attempts = 5;

                        while (true)
                        {
                            Console.CursorTop = 2; 

                            if (FT4222_GPIO_Read(ftHandle_GPIO, GPIO_Port.GPIO_PORT3, ref value) == FT4222_STATUS.FT4222_OK)
                            {
                                if (value)
                                {
                                    ft42Status = FT4222_SPIMaster_SingleReadWrite(ftHandle_SPI, ref rxADCrawData1[0], ref tx[0], (ushort)rxADCrawData1.Length, ref sizeTransferred, true);
                                    TotalSampleCount++;
                                
                                    for (int i = 0; i < 8; i++)
                                    {
                                        // Identifying the channel number from the header byte
                                        int ch_id = (rxADCrawData1[4 * i] & 0x70) >> 4;


                                        if (i != ch_id)
                                        {
                                            ChannelIDError++;                   // total mismatches

                                            System.Threading.Thread.Sleep(10);

                                            // timestamps after each mismacth
                                            mismatch_mark2 = mismatch_mark1;           
                                            mismatch_mark1 = stopwatch.Elapsed;         
                                        

                                            // comparing timestamps to check if the batch of mismatches began
                                            if (mismatch_mark1.TotalSeconds - mismatch_mark2.TotalSeconds < 2)              
                                            {
                                                // mismatch batch - consecutive mismatches happen within 2 seconds

                                                ChM2 = ChannelIDError;          // storing number of mismatches if a batch of mismacthes begins
                                                mismatchBatch = true;           // mismatch within the batch is true
                                                TS2 = TotalSampleCount;         // storing total samples at this point 
                                            }
                                            else                                // consecutive mismatches happen within 2 seconds
                                            {
                                                // mismatch outside the batch of mismatches - consecutive mismatches happen over 2 seconds (markers and flags updated)

                                                mismatchBatch = false;          // random mismatch outside any batch of mismatch
                                                ChM1 = ChannelIDError;          // storing the number of mismacthes when it's not in the batch of mismatches
                                                TS1 = TotalSampleCount;         // storing total samples when random mismatch occurs
                                                mismatch_batch1 = stopwatch.Elapsed;        // updating marker timestamp 
                                            }

                                            // calculations for the batch of mismatch detected
                                            if (mismatchBatch && (ChM2 - ChM1 > 75))        //mismatch batch flag must be true fromt he above condition AND number of batch mismatches is above approx threshold of 75 mismatches
                                            {
                                                mismatchBatchOccurrence++;      // mismatch batch occurrence counter 
                                            
                                                 mismatch_batch2 = stopwatch.Elapsed;        // timestamp marker at the approx end of batch of mismatches

                                                SamplesDuringBatch = TS2 - TS1;             // total samples during this batch of mismatches

                                                batch_eff = (SamplesDuringBatch - (ChM2 - ChM1)) / SamplesDuringBatch * 100;           // approx good samples % 

                                                // reset flags and markers
                                                mismatchBatch = false;          
                                                ChM2 = 0;
                                                ChM1 = ChannelIDError;


                                                batch_period = mismatch_batch2.TotalSeconds - mismatch_batch1.TotalSeconds;         // storing the duration of this batch of mismatches

                                                System.Threading.Thread.Sleep(3);
                                            }


                                            // Ch0 PGA config
                                            ft42Status = FT4222_SPIMaster_SingleWrite(ftHandle_SPI, ref txBuf1[0], (ushort)txBuf.Length, ref sizeTransferred, true);

                                            goto read_data;

                                        }


                                        // ADCrawData buffer holds the raw data from 8 channels ------- EXTRACT data from this buffer
                                        ADCrawData[i] = rxADCrawData1[1 + 4 * i] * 256 * 256 + rxADCrawData1[2 + 4 * i] * 256 + rxADCrawData1[3 + 4 * i];


                                        // ------------------------------------Load cell count------------------------------
                                        //System.Threading.Thread.Sleep(2);
                                        // Clearing the previous results on the Console
                                        Console.WriteLine("Channel " + i + ":                     ");
                                        Console.CursorTop = Console.CursorTop - 1;
                                        // Displaying the results on the Console
                                        Console.SetCursorPosition(0, Console.CursorTop);
                                        if (ADCrawData[i] == 0)
                                            Console.WriteLine("Channel " + i + ": " + "N/A");
                                        else
                                            Console.WriteLine("Channel " + i + ": " + ADCrawData[i]);
                                        //------------------------------------------------------------------------------------

                                        // out of range error condition
                                        //if ((ADCrawData[i] <  ) || (ADCrawData[i] >  ))
                                        //{
                                        //    OutOfRangeError++;
                                        //}

                                    }

                                    efficiency = 100 * (TotalSampleCount - ChannelIDError - OutOfRangeError) / TotalSampleCount;        // calculating over all efficiency (concentrated on mismatch)
                                
                                    // ---------------------------------EFFICIENCY------------------------------------------
                                    // change the console width between 155 and 160 for easier monitoring
                                    Console.WriteLine("TotalSamples:" + TotalSampleCount + "  TotalMismatches: " + ChannelIDError + "  (BatchOccurrence: "+ mismatchBatchOccurrence + "  BatchDuration:" + String.Format(" {0:0}", batch_period) + "  SamplesDuringBatch: " + SamplesDuringBatch + "  BatchEfficiency:" + String.Format(" {0:00}", batch_eff) + "%)" + /*"  OutOfRangeError:" + OutOfRangeError +*/ "   OverallSuccess:" + String.Format(" {0:00.00}", efficiency) + "%");
                                    //                 Total samples during the run                 total mismatch count                batch of mismatches count                       duration of this batch of mismatches                                total samples during this batch             % of good samples during this batch                         total out of range error count                      overall % of good reads                             
                                    //--------------------------------------------------------------------------------------

                                    // Ch0 PGA config
                                    ft42Status = FT4222_SPIMaster_SingleWrite(ftHandle_SPI, ref txBuf1[0], (ushort)txBuf.Length, ref sizeTransferred, true);
                                
                                }
                            }



                            System.Threading.Thread.Sleep(4);
                            continue;
                        }                        
                    }
                    else
                    {
                        attempts--;
                        System.Threading.Thread.Sleep(400);
                    }
                }
            }
        }
    }
}
